/***************************************************************************

    PROGRAM: Autotest.c

    PURPOSE: Autotest is a capture/playback tool, implemented as a simple
         Windows application.  The user records a sequence of keystrokes
         or mouse messages by picking the Start Record menu option.
         Autotest calls the Tester routine, in the TESTER.DLL to begin
         recording.  Tester installs a WH_JOURNALRECORD hook and keyboard
         and mouse events are saved in global memory.  Autotest calls the
         Tester routine again, when the user picks the Start Playback
         menu option.  Tester installs a WH_JOURNALPLAYBACK hook to
         playback the previously recorded events.

******************************************************************************/

#include "windows.h"
#include "autotest.h"
#include "tester.h"

HANDLE hInst;
HWND   hWindow;
HCURSOR hCursor;

static  char szFileName[128];
char str[255];

HANDLE hHourGlass;                        /* handle to hourglass cursor    */
HANDLE hSaveCursor;                       /* current cursor handle         */
int hFile;                                /* file handle                   */
OFSTRUCT OfStruct;                        /* information from OpenFile()   */

RECT Rect;                                /* dimension of the client window */
HWND hWnd;

static PGMSTATUS       Autotest_status;
static TESTRESULT      AutotestResult;

static BOOL            TraceOnFlag = FALSE;
static GLOBALHANDLE    hMemTestEvents;

/****************************************************************************

    FUNCTION: WinMain(HANDLE, HANDLE, LPSTR, int)

    PURPOSE: calls initialization function, processes message loop

****************************************************************************/

int PASCAL WinMain(HANDLE hInstance, HANDLE hPrevInstance, LPSTR lpCmdLine,
                   int nCmdShow)
{
    MSG msg;

    if (!hPrevInstance)
       if (!InitApplication(hInstance))
          return (FALSE);

    if (!InitInstance(hInstance, nCmdShow))
       return (FALSE);

    while (GetMessage(&msg, NULL, NULL, NULL))
       {
       TranslateMessage(&msg);
       DispatchMessage(&msg);
       }
    return (msg.wParam);
}


/****************************************************************************

    FUNCTION: InitApplication(HANDLE)

    PURPOSE: Initializes window data and registers window class

****************************************************************************/

BOOL InitApplication(HANDLE hInstance)
{
    WNDCLASS  wc;

    wc.style         = CS_DBLCLKS;          /* double-click messages */
    wc.lpfnWndProc   = MainWndProc;
    wc.cbClsExtra    = 0;
    wc.cbWndExtra    = 0;
    wc.hInstance     = hInstance;
    wc.hIcon         = LoadIcon(hInstance, "AutotestIcon");
    wc.hCursor       = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground = GetStockObject(WHITE_BRUSH);
    wc.lpszMenuName  = "AutotestMenu";
    wc.lpszClassName = "AutotestWClass";

    return (RegisterClass(&wc));
}


/****************************************************************************

    FUNCTION:  InitInstance(HANDLE, int)

    PURPOSE:  Saves instance handle and creates main window

****************************************************************************/

BOOL InitInstance(HANDLE hInstance, int nCmdShow)
{
    HWND            hWnd;


    hInst = hInstance;


    hWnd = CreateWindow(
              "AutotestWClass",
              "AUTOTEST",
              WS_OVERLAPPEDWINDOW,  /* horz & vert scroll bars */
              (GetSystemMetrics(SM_CXSCREEN)/2),
              0,
              (GetSystemMetrics(SM_CXSCREEN)/2),
              (GetSystemMetrics(SM_CYCAPTION)*3),
              NULL,
              NULL,
              hInstance,
              NULL
              );

    if (!hWnd)
       return (FALSE);

    szFileName[0] = '\0';

    ShowWindow(hWnd, nCmdShow);
    UpdateWindow(hWnd);
    return (TRUE);

}


/****************************************************************************

    FUNCTION: MainWndProc(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages

    MESSAGES:

        WM_COMMAND  - application menu (About dialog box)
        IDM_TRACEON - turns on trace mode, which causes the Tester
                      routine to write out recorded event messages,
                      and played-back event messages to trace files.
       IDM_TRACEOFF - turns off trace mode.  No trace data is saved.
        IDM_RECORD  - turns on record mode, causes the Tester routine
                      to be invoked.  Tester installs a WH_JOURNALRECORD
                      hook and records user event messages.
        IDM_STOPREC - calls the Tester routine to turn off record mode
                      and to de-install the WH_JOURNALRECORD hook.
        IDM_REPLAY  - turns on playback mode, the Tester routine is
                      invoked.  Tester installs a WH_JOURNALPLAYBACK hook
                      which plays back previously recorded messages.

        WM_DESTROY  - destroy window


****************************************************************************/

long FAR PASCAL MainWndProc(HWND hWnd, unsigned message, WORD wParam,
                            LONG lParam)
{
    FARPROC lpProcAbout;

    BOOL    RecordingFlag = FALSE;
    BOOL    ReplayingFlag = FALSE;


    switch (message)
       {
       case WM_INITMENU:
          switch (Autotest_status)
             {
             case NOT_STARTED_RECORDING:
                EnableMenuItem(GetMenu(hWnd), IDM_RECORD,
                               MF_BYCOMMAND | MF_ENABLED);
                EnableMenuItem(GetMenu(hWnd), IDM_STOPREC,
                               MF_BYCOMMAND | MF_GRAYED);
                EnableMenuItem(GetMenu(hWnd), IDM_REPLAY,
                               MF_BYCOMMAND | MF_GRAYED);
                break;

             case RECORDING_ON:
                EnableMenuItem(GetMenu(hWnd), IDM_RECORD,
                               MF_BYCOMMAND | MF_GRAYED);
                EnableMenuItem(GetMenu(hWnd), IDM_STOPREC,
                               MF_BYCOMMAND | MF_ENABLED);
                EnableMenuItem(GetMenu(hWnd), IDM_REPLAY,
                               MF_BYCOMMAND | MF_GRAYED);
                break;

             case RECORDING_COMPLETED:
                EnableMenuItem(GetMenu(hWnd), IDM_RECORD,
                               MF_BYCOMMAND | MF_GRAYED);
                EnableMenuItem(GetMenu(hWnd), IDM_STOPREC,
                               MF_BYCOMMAND | MF_GRAYED);
                EnableMenuItem(GetMenu(hWnd), IDM_REPLAY,
                               MF_BYCOMMAND | MF_ENABLED);
                break;

             case REPLAYING_ON:
                EnableMenuItem(GetMenu(hWnd), IDM_RECORD,
                               MF_BYCOMMAND | MF_GRAYED);
                EnableMenuItem(GetMenu(hWnd), IDM_STOPREC,
                               MF_BYCOMMAND | MF_GRAYED);
                EnableMenuItem(GetMenu(hWnd), IDM_REPLAY,
                               MF_BYCOMMAND | MF_GRAYED);
                break;

             case REPLAYING_COMPLETED:
                EnableMenuItem(GetMenu(hWnd), IDM_RECORD,
                               MF_BYCOMMAND | MF_ENABLED);
                EnableMenuItem(GetMenu(hWnd), IDM_STOPREC,
                               MF_BYCOMMAND | MF_GRAYED);
                EnableMenuItem(GetMenu(hWnd), IDM_REPLAY,
                               MF_BYCOMMAND | MF_ENABLED);
                break;
             }

          if (!TraceOnFlag)
             {
             EnableMenuItem(GetMenu(hWnd), IDM_TRACEON,
                            MF_BYCOMMAND | MF_ENABLED);
             EnableMenuItem(GetMenu(hWnd), IDM_TRACEOFF,
                            MF_BYCOMMAND | MF_GRAYED);
             }
          else
             {
             EnableMenuItem(GetMenu(hWnd), IDM_TRACEOFF,
                            MF_BYCOMMAND | MF_ENABLED);
             EnableMenuItem(GetMenu(hWnd), IDM_TRACEON,
                            MF_BYCOMMAND | MF_GRAYED);
             }
          break;

       case WM_COMMAND:
          hWindow = hWnd;
          switch (wParam)
             {
             case IDM_TRACEON:
                GetProfileString("autotest","RecordTracefile","rectrace.txt",
                                 szFileName, FILENAMELEN);
                if (-1 == (hFile = OpenFile(szFileName, (LPOFSTRUCT) &OfStruct,
                                           OF_CREATE | OF_PROMPT | OF_CANCEL)))
                   {
                   TraceOnFlag = FALSE;
                   wsprintf(str, "Cannot open trace file = %s", szFileName);
                   MessageBox(hWnd, str, "ERROR", MB_APPLMODAL | MB_OK |
                              MB_ICONHAND);
                   }
                else
                   {
                   TraceOnFlag = TRUE;
                   /* Check the filename for syntax errors Use default
                      if errors exist  */
                   _lclose(hFile);
                   }
                break;

             case IDM_TRACEOFF:
                TraceOnFlag = FALSE;
                szFileName[0] = '\0';
                break;

             case IDM_ABOUT:
                lpProcAbout = MakeProcInstance(About, hInst);
                DialogBox(hInst,
                         "AboutBox",
                          hWnd,
                          lpProcAbout);
                FreeProcInstance(lpProcAbout);
                break;

             case IDM_RECORD:
                if (Autotest_status != NOT_STARTED_RECORDING)
                   {
                   /*  Clear out the previous test script  */
                   if (hMemTestEvents != NULL)
                      {
                      hMemTestEvents = GlobalFree(hMemTestEvents);
                      }
                   }
                Autotest_status = RECORDING_ON;
                AutotestResult = Tester(IDD_STARTRECORD, hMemTestEvents,
                                        hWnd, IDD_HALTACTIVITY,
                                        TraceOnFlag, (LPSTR)szFileName);
                break;

             case IDM_STOPREC:
                if (Autotest_status == RECORDING_ON)
                   {
                   hCursor = SetCursor(LoadCursor(NULL, IDC_WAIT));
                   ShowCursor(TRUE);
                   AutotestResult = Tester(IDD_STOPRECORD,
                   hMemTestEvents, hWnd, IDD_HALTACTIVITY,
                   TraceOnFlag, (LPSTR)szFileName);
                   ShowCursor(FALSE);
                   SetCursor(hCursor);
                   }
                break;

             case IDM_REPLAY:
                if ((Autotest_status == RECORDING_COMPLETED) ||
                    (Autotest_status == REPLAYING_COMPLETED))
                   /* Read in replaying parameters from win.ini */
                   {
                   GetProfileString("autotest","PlaybackTracefile",
                                    "plytrace.txt",szFileName, FILENAMELEN);
                   if (TraceOnFlag)
                      if (-1 == (hFile = OpenFile(szFileName,
                                (LPOFSTRUCT) &OfStruct,
                                 OF_CREATE | OF_PROMPT | OF_CANCEL)))
                         {
                         TraceOnFlag = FALSE;
                         wsprintf(str, "Cannot open trace file = %s",
                                  szFileName);
                         MessageBox(hWnd, str, "ERROR", MB_APPLMODAL | MB_OK |
                                    MB_ICONHAND);
                         }

                   if (hMemTestEvents != NULL)
                      {
                      /* Read in Timer values */
                      Autotest_status = REPLAYING_ON;
                      AutotestResult = Tester(IDD_STARTPLAY,
                                       hMemTestEvents, hWnd, IDD_HALTACTIVITY,
                                       TraceOnFlag, (LPSTR)szFileName);

                      }
                   }
                break;


             default:
                return (DefWindowProc(hWnd, message, wParam, lParam));
                break;
             }
          ErrorMessageHandler(AutotestResult);
          break;

       case IDD_HALTACTIVITY:
          if (Autotest_status == RECORDING_ON)
             Autotest_status = RECORDING_COMPLETED;
          else
             if (Autotest_status == REPLAYING_ON)
                Autotest_status = REPLAYING_COMPLETED;
             else
                MessageBox(hWnd, "Incorrect Test Status", "Autotest",
                           MB_SYSTEMMODAL | MB_OK);
          hMemTestEvents = wParam;
          ErrorMessageHandler((TESTRESULT)lParam);
          break;

       case WM_DESTROY:
          PostQuitMessage(0);
          break;

       default:
          return (DefWindowProc(hWnd, message, wParam, lParam));
          break;
       }
    return (NULL);
}


/****************************************************************************

    FUNCTION: About(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages for "About" dialog box

    MESSAGES:

       WM_INITDIALOG - initialize dialog box
       WM_COMMAND    - input received

****************************************************************************/

BOOL FAR PASCAL About(HWND hDlg, unsigned message, WORD wParam, LONG lParam)
{
   switch (message)
      {
      case WM_INITDIALOG:
         return (TRUE);

      case WM_COMMAND:
         if (wParam == IDD_OK)
            {
            EndDialog(hDlg, TRUE);
            return (TRUE);
            }
         break;
      }

   return (FALSE);
}
/****************************************************************************

    FUNCTION: ErrorMessageHandler(WORD)

    PURPOSE:  Processes errors return codes from the Tester routine.
              These return codes are passed through the lParam argument
              of the SendMessage call.

****************************************************************************/
void ErrorMessageHandler(WORD wReturnCode)
{
   char *szMessage = NULL;

   switch (wReturnCode)
      {
      case TEST_RECORDACTIVE:
         szMessage = "Tester already recording/playing.";
         break;

      case TEST_INACTIVE:
         szMessage = "Tester already stopped.";
         break;

      case TEST_NOMEMORY:
         szMessage = "Insufficient memory to begin recording.";
         break;

      case TEST_NOEVENTS:
         szMessage = "No events to playback.";
         break;

      case TEST_TOOMANYEVENTS:
         szMessage = "Too many events in event queue:  stop recording.";
         break;

      }
   if (szMessage != NULL)
       MessageBox(hWindow, szMessage, "Autotest", MB_OK | MB_ICONINFORMATION);

   return;
}

